#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include "lx.h"

/****************
 * Autoconfiguration
 ****************/

#if MI_CAN_BE_A_DAEMON

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>

int getdtablesize(void);
int open (const char *pathname, int flags, ...);
mode_t umask (mode_t mask);
pid_t setsid (void);

int demonize (const char*filename)
{ 
  int pid;
  int pid_file;
  int fd;
  char buf[16];

  if (!filename)
    return 0;
	
  /* Create a new thread, so the parent can exit immediately. */

  pid = fork ();
  if (pid < 0)
  {
    printf ("Failed to spawn daemon.\n");
    /* Error spawning daemon. */
    return 0;
  }
  else if (pid > 0)
  {
    /* Parent exits successfully. */
    exit (0);
  }
  
  /* Create a process group, so we won't be killed when our parent
     process group exits. */

  setsid ();		/* always succeeds for child */

  /* Close any left-over file descriptor */

  for (fd=3;fd < 255;fd++) {
    close(fd);
  }

  /* Write the process ID to a file, taking care that there is
     no other mapper running and holding a lock on this file. */
      
  pid_file = open (filename, O_RDWR | O_CREAT, 0644);
  if (pid_file < 0)
  {
    printf ("could not open %s:", filename);
    perror ("");
    return 0;
  }
  if (lockf (pid_file, F_TLOCK, 0) < 0)
  {
    /* We cannot lock the file, so there must be another mapper
       running. */
    printf ("could not lock %s:", filename);
    perror ("");
    return 0;
  }
  sprintf (buf, "%d\n", (int)getpid ());
  if (write (pid_file, buf, strlen (buf)) == -1)
  {
    printf ("could not write PID to %s:", filename);
    perror ("");
    return 0;
  }
  
  /* Close any open files that might be attached to the controlling
     terminal.  Not doing this is know to prevent the mapper from
     being spawned via ssh. */

  if ((fd = open("/dev/null", O_RDWR, 0)) != -1)
  {
    struct stat s;
    (void)dup2(fd, STDIN_FILENO);
    if (fstat(STDOUT_FILENO, &s) || !S_ISREG(s.st_mode))
      (void)dup2(fd, STDOUT_FILENO);
    if (fstat(STDERR_FILENO, &s) || !S_ISREG(s.st_mode))
      (void)dup2(fd, STDERR_FILENO);
    if (fd > 2)
    {
      (void)close(fd);
    }
  }
  return 1;
}

#endif /* CAN_DAEMON */

#if MI_OS_WINNT
void windows_service(void);
#endif

/****************
 * End of autoconfiguration
 ****************/

const char*map_filenames [MI_MY_SIZE];
const char*out_filename;
const char*daemon_pid_filename;
int mi_verbose_flag;
FILE*out_fp;

static void signal_handler (int sig)
{
  (void) lx_set_verbosity(-1);
}

int lx_set_verbosity(int state)
{
  int old_flag = mi_verbose_flag;

  mi_verbose_flag = (state == -1) ? !mi_verbose_flag : state;

  if (out_filename)
  {
    if (mi_verbose_flag && !out_fp)
    {
      out_fp = fopen (out_filename, "w");
    }
    else if (!mi_verbose_flag && out_fp)
    {
      fclose (out_fp);
      out_fp = 0;
    }
  }

  return old_flag;
}

/* Return nonzero iff arg matches prefix.  Store a pointer to the remaining
   arg at match, if match is nonzero. */

static int match (const char *arg, const char *prefix, const char **match)
{
  size_t prefix_len;

  prefix_len = strlen (prefix);
  if (!strncmp (arg, prefix, prefix_len))
  {
    if (match)
      *match = arg + prefix_len;
    return 1;
  }
  return 0;
}

static int short_match (const char *arg, const char *prefix, const char **match,
			int *argc, char **argv[])
{
  size_t prefix_len;

  prefix_len = strlen (prefix);
  if (!strncmp (arg, prefix, prefix_len))
  {
    if (*argc > 1)
    {
      (*argc)--;
      (*argv)++;
      *match = **argv;
      return 1;
    }
  }
  return 0;
}

/* Print program usage information. */

void
usage (const char *program_name)
{
  printf
    ("usage: %s [options]\n"
     "valid options include:\n"
     "  --daemon-pid-file=<path>\n"
     "    Run as a daemon, storing daemon PID at <path>.\n"
     "  --help\n"
     "    print this message and exit\n"
     "  --level=<level>\n"
     "    mapper level (default 1, 0 for never map).\n"
     "  --map-file-0=<path>\n"
     "    Store map file for port 0 at <path>.\n"
     "  --map-file-1=<path>\n"
     "    Store map file for port 1 at <path>.\n"
     "  --pause\n"
     "    for mute find_xbars: pause all other mappers\n"
     "  --map-once\n"
     "    terminate after mapping is complete everywhere.\n"
     "  -B <Board>,\n" 
     "  -b <board>,\n" 
     "  --unit=<board>\n"
     "    board number.\n"
     " --routes=[careful | shortest | updown | <basename>]\n"
     "    select routing algorithm (default careful)\n"
     "    if <basename> specified:\n"
     "      <basename>.routes[0,1] is an optimal routes file,\n"
     "      <basename>.map[0,1] is the corresponding map file,\n"
     "      use routes as close to these as possible\n"
     " --xbar-verify=[none | clos | updown | all]\n"
     "    select xbar verify (default clos)\n"
     "  --seek-size=<level>\n"
     "    seek size.\n"
     "  --active-tries=<small number>\n"
     "    retry count for active mapping.\n"
     "  -v,\n"
     "  --verbose\n"
     "    verbose output.\n"
     "  --verbose-file=<path>\n"
     "    write verbose output to <path>\n"
     "  --thrash=<period>\n"
     "    another debugging feature. keep remapping.\n"
     "  --extra-hosts=<count>\n"
     "    a debugging feature. make artificially inflated map.\n",

     program_name);
}

struct lx_t lx;

int main (int argc, char*argv[])
{
  int unit = 0;
  const char*s;
  char*program_name = argv [0];
  int level = 1;
  int routing_flag = LX_FLAG_ROUTING_CAREFUL;
  int xbar_verify_flag = LX_FLAG_XBAR_VERIFY_CLOS;
  int check_xbars = 1;
  int map_once = 0;
  int extra_hosts = 0;
  int service = 0;
  int thrash = 0;
  int pause = 0;
  int seek_size = LX_SEEK_SIZE;
  int active_tries = LX_ACTIVE_TRIES;
  char *route_filebase = NULL;

  int i;
  
  for (i = 0; i < MI_MY_SIZE; i++)
    map_filenames [i] = 0;
  
  /* Parse command-line parameters. */

  for (argc--, argv++; argc; argc--, argv++)
  {
    
    if (MI_CAN_BE_A_DAEMON && match (*argv, "--daemon-pid-file=", &daemon_pid_filename))
    {
      printf ("Storing daemon PID in file \"%s\"\n", daemon_pid_filename);
      continue;
    }
    if (!strcmp (*argv, "--help"))
    {
      usage (program_name);
      exit (0);
    }
    if (match (*argv, "--map-file-0=", &map_filenames [0]))
      continue;
    if (match (*argv, "--map-file-1=", &map_filenames [1]))
      continue;
    if (match (*argv, "--verbose-file=", &out_filename))
      continue;
    if (match (*argv, "--unit=", &s)
	|| short_match (*argv, "-B", &s, &argc, &argv)
	|| short_match (*argv, "-b", &s, &argc, &argv))
    {
      unit = atoi (s);
      continue;
    }
    if (match (*argv, "--level=", &s))
    {
      level = atoi (s);
      continue;
    }
    if (match (*argv, "--seek-size=", &s))
    {
      seek_size = atoi (s);
      continue;
    }
    if (match (*argv, "--active-tries=", &s))
    {
      active_tries = atoi (s);
      continue;
    }
    if (match (*argv, "--routes=", &s))
    {
      if (!strcmp (s, "shortest"))
	routing_flag = LX_FLAG_ROUTING_SHORTEST_PATH;
      else if (!strcmp (s, "updown"))
	routing_flag = LX_FLAG_ROUTING_UP_DOWN;
      else if (!strcmp (s, "careful"))
	routing_flag = LX_FLAG_ROUTING_CAREFUL;
      else
      {
	routing_flag = LX_FLAG_ROUTING_SHORTEST_PATH;
	route_filebase = (char *) malloc (strlen (s)+1);
	insist (route_filebase != NULL);
	strcpy (route_filebase, s);
      }
      continue;
    }
    if (match (*argv, "--xbar-verify=", &s))
    {
      if (!strcmp (s, "none"))
	xbar_verify_flag = LX_FLAG_XBAR_VERIFY_NONE;
      else if (!strcmp (s, "clos"))
	xbar_verify_flag = LX_FLAG_XBAR_VERIFY_CLOS;
      else if (!strcmp (s, "updown"))
	xbar_verify_flag = LX_FLAG_XBAR_VERIFY_UP_DOWN;
      else if (!strcmp (s, "all"))
	xbar_verify_flag = LX_FLAG_XBAR_VERIFY_ALL;
      else
      {
	usage (program_name);
	exit (0);
	
      }
      continue;
    }
    if (match (*argv, "--extra-hosts=", &s))
    {
      extra_hosts = atoi (s);
      continue;
    }
    if (match (*argv, "--thrash=", &s))
    {
      thrash = atoi (s);
      continue;
    }
    if (!strcmp (*argv, "--no-xbar-check"))
    {
      check_xbars = 0;
      continue;
    }
    if (!strcmp (*argv, "--map-once"))
    {
      map_once = 1;
      continue;
    }
    if (!strcmp (*argv, "--pause"))
    {
      pause = 1;
      continue;
    }
    if (!strcmp (*argv, "--verbose") || !strcmp (*argv, "-v"))
    {
      mi_verbose_flag = 1;
      continue;
    }
    if (!strcmp (*argv, "--service"))
    {
      service = 1;
      continue;
    }

    usage (program_name);
    printf ("Unrecognized option: \"%s\"\n", *argv);
    exit (1);
  }  

  if (mi_verbose_flag && !daemon_pid_filename && out_filename)
  {
    mi_verbose_flag = 0;
    signal_handler (0);
  }
  
#if MI_CAN_BE_A_DAEMON
  /* Become a daemon, if necessary. */
  if (daemon_pid_filename)
  {
    if (!demonize (daemon_pid_filename))
      return 1;
  }
#endif

#if !MI_OS_WINNT
  signal (SIGHUP, signal_handler);
#endif

#if MI_OS_WINNT
  if (service)
    windows_service ();
  else
#endif
  {

    lx.level = level;
    lx.map_once = map_once;
    lx.xbar_verify_flag = xbar_verify_flag;
    lx.routing_flag = routing_flag;
    lx.pause = pause;
    lx.extra_hosts = extra_hosts;
    lx.thrash = thrash;
    lx.seek_size = seek_size;
    lx.active_tries = active_tries;
    lx.route_filebase = route_filebase;
    
    mi_go (&lx, unit);
  }
  return 0;
  except: return 1;
}
